//  Copyright (c) 2009 Karl Blomster
//
//  Permission is hereby granted, free of charge, to any person obtaining a copy
//  of this software and associated documentation files (the "Software"), to deal
//  in the Software without restriction, including without limitation the rights
//  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//  copies of the Software, and to permit persons to whom the Software is
//  furnished to do so, subject to the following conditions:
//
//  The above copyright notice and this permission notice shall be included in
//  all copies or substantial portions of the Software.
//
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//  THE SOFTWARE.


#include <shlobj.h>
#include <tchar.h>
#include <time.h>
#include <string>
#include "assrender.h"


#define _R(c)  ((c)>>24)
#define _G(c)  (((c)>>16)&0xFF)
#define _B(c)  (((c)>>8)&0xFF)
#define _A(c)  ((c)&0xFF)


char cur_dll_path[MAX_PATH]; // hack
int		loglevel;
FILE	*log;

// the point of this win32 api exercise is to get the path of the dll,
// so we can try to load a fontconfig configuration file from a path
// relative to the dll's path.
DWORD GetModulePath(HMODULE hModule, LPTSTR pszBuffer, DWORD dwSize) {
	DWORD dwLength = GetModuleFileName(hModule, pszBuffer, dwSize);
	if (dwLength) {
		while(dwLength && pszBuffer[dwLength] != _T('\\')) {
			dwLength--;
		}
		if (dwLength)
			pszBuffer[dwLength+1] = _T('\000');
	}
	return dwLength;
}


BOOL APIENTRY DllMain(HMODULE hModule, DWORD fdwReason, LPVOID lpReserved ) {
	GetModulePath(hModule, cur_dll_path, MAX_PATH);
	return TRUE;
}


AssRender::AssRender(PClip _child, const char *_filename, ASS_Hinting hints, double scale, 
	const char *_charset, int loglevel, std::string logfilename, IScriptEnvironment* env) :
GenericVideoFilter(_child) {
	// this is pretty faggy but whatever
	if (!vi.IsRGB32())
		env->ThrowError("AssRender: only RGB32 input is supported");

	lp.loglevel = loglevel;
	if (loglevel >= 0) {
		lp.log = fopen(logfilename.c_str(), "a");
		if (!lp.log)
			lp.loglevel = -1;
	}
	else
		lp.log = NULL;

	if(!InitLibass(hints, scale))
		env->ThrowError("AssRender: failed to initialize libass");

	char filename[MAX_PATH]; // libass wants non-const
	strcpy(filename, _filename);
	char charset[128]; // 128 bytes ought to be enough for anyone
	strcpy(charset, _charset);

	t = ass_read_file(al, filename, charset);
	if (!t)
		env->ThrowError("AssRender: could not read %s", filename);
}


AssRender::~AssRender() {
	ass_free_track(t);
	ass_renderer_done(ar);
	ass_library_done(al);
	if (lp.log)
		fclose(lp.log);
}


bool AssRender::InitLibass(ASS_Hinting hints, double scale) {
	al = ass_library_init();
	if (!al)
		return false;

	ass_set_message_cb(al, &AssRender::LoggingCallback, &lp);

	//char tmp[MAX_PATH];
	//SHGetFolderPath(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, tmp);

	// not needed?
	//ass_set_fonts_dir(al, tmp);
	ass_set_extract_fonts(al, 0);
	ass_set_style_overrides(al, NULL);

	ar = ass_renderer_init(al);
	if (!ar)
		return false;

	ass_set_frame_size(ar, vi.width, vi.height);
	//ass_set_margins(ar, 0, 0, 0, 0);
	//ass_set_use_margins(ar, 0);
	//ass_set_aspect_ratio(ar,);  // todo: implement this
	ass_set_font_scale(ar, scale);
	ass_set_hinting(ar, hints);

	std::string path(cur_dll_path);
	path.append("fontconfig\\fonts.conf");
	ass_set_fonts(ar, "Arial", "Sans", 1, path.c_str(), 1);

	return true;
}


void AssRender::LoggingCallback(int level, const char *fmt, va_list args, void *data) {
	logprivate *lp = (logprivate *)data;
	if (lp->loglevel < level)
		return;
	
	char time[10];
	_strtime(time);

	std::string out("[");
	out = out + time + "] " + fmt + '\n';
	vfprintf(lp->log, out.c_str(), args);
	fflush(lp->log);
}



PVideoFrame __stdcall AssRender::GetFrame(int n, IScriptEnvironment* env) {
	PVideoFrame src = child->GetFrame(n, env);
	PVideoFrame dst = env->NewVideoFrame(vi);
	
	const unsigned char* srcp = src->GetReadPtr();
	unsigned char* dstp = dst->GetWritePtr();
	
	const int dst_pitch = dst->GetPitch();
	const int dst_w = dst->GetRowSize();
	const int dst_h = dst->GetHeight();

	const int src_pitch = src->GetPitch();
	const int src_w = src->GetRowSize();
	const int src_h = src->GetHeight();

	env->BitBlt(dstp, dst_pitch, srcp, src_pitch, dst_w, dst_h);

	// it's a casting party!
	int64_t ts = (int64_t)n * (int64_t)1000 * (int64_t)vi.fps_denominator / (int64_t)vi.fps_numerator;

	ASS_Image *img = ass_render_frame(ar, t, ts, NULL);

	// this here loop shamelessly stolen from aegisub's subtitles_provider_libass.cpp
	// and slightly modified to render upside-down
	while (img) {
		if (img->w == 0 || img->h == 0)
			continue;

		unsigned int a = 255 - ((unsigned int)_A(img->color));
		unsigned int r = (unsigned int)_R(img->color);
		unsigned int g = (unsigned int)_G(img->color);
		unsigned int b = (unsigned int)_B(img->color);

		int dst_delta = dst_pitch + img->w*4;
		const unsigned char *src_map = img->bitmap;
		// Move destination pointer to the bottom right corner of the bounding
		// box that contains the current overlay bitmap.
		// Remember that avisynth rgb32 bitmaps are upside down, hence we need
		// to render upside down.
		unsigned char *dst_map = dstp + (dst_pitch * (dst_h - img->dst_y - 1)) + img->dst_x * 4;
		unsigned int k, ck, t;

		for (int y=0; y < img->h; y++) {
			for (int x = 0; x < img->w; ++x) {
				k = ((unsigned)src_map[x]) * a / 255;
				ck = 255 - k;
				t = *dst_map;
				*dst_map++ = (k*b + ck*t) / 255;
				t = *dst_map;
				*dst_map++ = (k*g + ck*t) / 255;
				t = *dst_map;
				*dst_map++ = (k*r + ck*t) / 255;
				dst_map++;
			}

			dst_map -= dst_delta;	// back up one scanline (remembering we're rendering bottom-to-top)
			src_map += img->stride;	// advance source one scanline
		}

		img = img->next;
	}


	return dst;
}


AVSValue __cdecl Create_AssRender(AVSValue args, void* user_data, IScriptEnvironment* env) {
	PClip c			= args[0].AsClip();
	const char *f	= args[1].AsString();
	int h			= args[2].AsInt(2);
	double scale	= args[3].AsFloat(1.0);
	const char *cs	= args[4].AsString("UTF-8");
	int loglevel	= args[5].AsInt(-1);
	const char *log = args[6].AsString("");
	ASS_Hinting hinting;

	if (!args[1].Defined())
		env->ThrowError("AssRender: no input file specified");

	switch (h) {
		case 0: hinting = ASS_HINTING_NONE; break;
		case 1: hinting = ASS_HINTING_LIGHT; break;
		case 2: hinting = ASS_HINTING_NORMAL; break;
		case 3: hinting = ASS_HINTING_NATIVE; break;
		default:
			env->ThrowError("AssRender: invalid hinting mode");
	}

	std::string logfile(log);
	if (logfile.empty() && loglevel >= 0)
		env->ThrowError("AssRender: you must specify a log file");

    return new AssRender(c, f, hinting, scale, cs, loglevel, logfile, env);  
}


extern "C" __declspec(dllexport) const char* __stdcall AvisynthPluginInit2(IScriptEnvironment* env) {
    env->AddFunction("AssRender", "c[file]s[hinting]i[scale]f[charset]s[loglevel]i[logfile]s", Create_AssRender, 0);

	return "AssRender: draws .asses better and faster than ever before";
}
